<?php
/*
Plugin Name: SM SEO Optimizer Pro
Plugin URI: https://smarttoolhub.digital/
Description: AI-assisted SEO optimizer: meta tags, keywords, sitemap, IndexNow, readability, alt checks, and SEO score.
Version: 1.0.0
Author: SmartToolHub
Author URI: https://smarttoolhub.digital/
License: GPLv2 or later
Text Domain: sm-seo-optimizer-pro
*/

if (!defined('ABSPATH')) { exit; }

define('SMSEO_VERSION', '1.0.0');
define('SMSEO_PLUGIN_FILE', __FILE__);
define('SMSEO_PLUGIN_DIR', plugin_dir_path(__FILE__));

require_once SMSEO_PLUGIN_DIR . 'includes/util.php';

class SMSEO_Optimizer_Pro {
    public function __construct() {
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('add_meta_boxes', [$this, 'add_meta_boxes']);
        add_action('save_post', [$this, 'on_save_post'], 10, 3);
        add_action('wp_head', [$this, 'output_meta_tags'], 1);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin']);
        register_activation_hook(SMSEO_PLUGIN_FILE, [$this, 'on_activate']);
        register_deactivation_hook(SMSEO_PLUGIN_FILE, [$this, 'on_deactivate']);
    }

    public function enqueue_admin($hook) {
        if (strpos($hook, 'sm-seo-optimizer') !== false) {
            wp_enqueue_style('smseo-admin', plugins_url('assets/css/admin.css', SMSEO_PLUGIN_FILE), [], SMSEO_VERSION);
            wp_enqueue_script('smseo-admin', plugins_url('assets/js/admin.js', SMSEO_PLUGIN_FILE), [], SMSEO_VERSION, true);
        }
    }

    public function admin_menu() {
        add_menu_page('SM SEO', 'SM SEO', 'manage_options', 'sm-seo-optimizer', [$this, 'render_admin'], 'dashicons-chart-area', 85);
        add_submenu_page('sm-seo-optimizer', 'Settings', 'Settings', 'manage_options', 'sm-seo-settings', [$this, 'render_settings']);
    }

    public function render_admin() {
        if (!current_user_can('manage_options')) { return; }
        $nonce = wp_create_nonce('smseo_scan');
        $summary = smseo_get_site_summary();
        echo '<div class="wrap"><h1>SM SEO Optimizer Pro</h1>';
        echo '<p>Indexed pages (approx): ' . esc_html($summary['indexed_estimate']) . '</p>';
        echo '<p>Average SEO Score: ' . esc_html($summary['avg_score']) . '</p>';
        echo '<form method="post"><input type="hidden" name="smseo_action" value="scan"/><input type="hidden" name="_wpnonce" value="' . esc_attr($nonce) . '"/><button class="button button-primary">Scan Broken Links</button></form>';
        if (isset($_POST['smseo_action']) && $_POST['smseo_action'] === 'scan' && wp_verify_nonce($_POST['_wpnonce'] ?? '', 'smseo_scan')) {
            $report = smseo_scan_broken_links();
            echo '<h2>Broken Links</h2><ul>';
            foreach ($report as $r) { echo '<li>' . esc_html($r['url']) . ' — ' . esc_html($r['status']) . '</li>'; }
            echo '</ul>';
        }
        echo '</div>';
    }

    public function render_settings() {
        if (!current_user_can('manage_options')) { return; }
        if (isset($_POST['smseo_save']) && check_admin_referer('smseo_save_settings')) {
            update_option('smseo_indexnow_key', sanitize_text_field($_POST['smseo_indexnow_key'] ?? ''));
            update_option('smseo_enable_indexnow', isset($_POST['smseo_enable_indexnow']) ? '1' : '0');
            echo '<div class="notice notice-success"><p>Settings saved.</p></div>';
        }
        $key = get_option('smseo_indexnow_key', '');
        $enabled = get_option('smseo_enable_indexnow', '0');
        echo '<div class="wrap"><h1>SM SEO Settings</h1><form method="post">';
        wp_nonce_field('smseo_save_settings');
        echo '<table class="form-table"><tr><th>IndexNow Key</th><td><input type="text" name="smseo_indexnow_key" value="' . esc_attr($key) . '" class="regular-text"/></td></tr>';
        echo '<tr><th>Enable IndexNow</th><td><label><input type="checkbox" name="smseo_enable_indexnow" ' . checked('1', $enabled, false) . ' /> Enable instant ping</label></td></tr>';
        echo '</table><p><button class="button button-primary" name="smseo_save" value="1">Save Settings</button></p></form></div>';
    }

    public function add_meta_boxes() {
        foreach (['post','page'] as $pt) {
            add_meta_box('smseo_meta', 'SM SEO Optimizer', [$this, 'render_meta_box'], $pt, 'side', 'default');
        }
    }

    public function render_meta_box($post) {
        $score = (int) get_post_meta($post->ID, '_smseo_score', true);
        $readability = (float) get_post_meta($post->ID, '_smseo_readability', true);
        $keywords = (array) get_post_meta($post->ID, '_smseo_keywords', true);
        $missingAlts = (int) get_post_meta($post->ID, '_smseo_missing_alts', true);
        echo '<p><strong>SEO Score:</strong> ' . esc_html($score) . '/100</p>';
        echo '<p><strong>Readability (Flesch):</strong> ' . esc_html($readability) . '</p>';
        if (!empty($keywords)) { echo '<p><strong>Recommended Keywords:</strong><br/>' . esc_html(implode(', ', array_slice($keywords,0,8))) . '</p>'; }
        echo '<p><strong>Missing Image Alts:</strong> ' . esc_html($missingAlts) . '</p>';
        echo '<p class="description">Meta title/description auto-generate on save.</p>';
    }

    public function on_save_post($post_id, $post, $update) {
        if (wp_is_post_revision($post_id) || (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)) { return; }
        if (!in_array($post->post_type, ['post','page'], true)) { return; }
        if (!current_user_can('edit_post', $post_id)) { return; }

        $content = wp_strip_all_tags($post->post_content);
        $title = wp_strip_all_tags($post->post_title);

        $metaTitle = smseo_generate_meta_title($title, $content);
        $metaDesc = smseo_generate_meta_description($content);
        update_post_meta($post_id, '_smseo_title', $metaTitle);
        update_post_meta($post_id, '_smseo_description', $metaDesc);

        $keywords = smseo_extract_keywords($content);
        update_post_meta($post_id, '_smseo_keywords', $keywords);

        $readability = smseo_flesch_score($content);
        update_post_meta($post_id, '_smseo_readability', $readability);

        $missingAlts = smseo_count_missing_alts($post->post_content);
        update_post_meta($post_id, '_smseo_missing_alts', $missingAlts);

        $score = smseo_compute_score([
            'title' => $title,
            'content' => $content,
            'metaTitle' => $metaTitle,
            'metaDesc' => $metaDesc,
            'keywords' => $keywords,
            'readability' => $readability,
            'missingAlts' => $missingAlts
        ]);
        update_post_meta($post_id, '_smseo_score', $score);

        smseo_regenerate_sitemap();
        if (get_option('smseo_enable_indexnow', '0') === '1') { smseo_indexnow_ping(get_permalink($post_id)); }
    }

    public function output_meta_tags() {
        if (!is_singular()) { return; }
        global $post; if (!$post) { return; }
        $mt = get_post_meta($post->ID, '_smseo_title', true);
        $md = get_post_meta($post->ID, '_smseo_description', true);
        if ($mt) { echo '<meta name="title" content="' . esc_attr($mt) . '" />' . "\n"; }
        if ($md) { echo '<meta name="description" content="' . esc_attr($md) . '" />' . "\n"; }
    }

    public function on_activate() {
        smseo_regenerate_sitemap();
    }

    public function on_deactivate() {
        // No cleanup for now
    }
}

new SMSEO_Optimizer_Pro();